import os
import json
import calendar
from datetime import date
from flask import Flask, request, render_template_string, jsonify

app = Flask(__name__)

DATA_DIR = os.path.join(os.path.dirname(__file__), "data")
os.makedirs(DATA_DIR, exist_ok=True)

TEMPLATE = r"""
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Calendar To-Do – {{ month_name }} {{ year }}</title>
    <style>
        body {
            font-family: sans-serif;
            margin: 0;
            padding: 0;
        }
        .topbar {
            padding: 10px;
            display: flex;
            align-items: center;
            gap: 10px;
            border-bottom: 1px solid #ccc;
        }
        .title {
            text-align: center;
            margin-top: 5px;
            margin-bottom: 10px;
            font-size: 1.3em;
        }
        .wrapper {
            padding: 10px;
        }
        .calendar {
            display: grid;
            grid-template-columns: repeat(7, 1fr);
            gap: 5px;
            width: 100%;
            box-sizing: border-box;
        }
        .weekday {
            text-align: center;
            font-weight: bold;
            padding: 5px 0;
            border-bottom: 1px solid #aaa;
        }
        .day-cell {
            border: 1px solid #ccc;
            display: flex;
            flex-direction: column;
            min-height: 120px;
            box-sizing: border-box;
        }
        .day-number {
            text-align: right;
            padding: 2px 4px;
            font-size: 0.8em;
            background: #f0f0f0;
            border-bottom: 1px solid #ddd;
        }
        .day-text {
            flex: 1;
            border: none;
            resize: vertical;
            width: 100%;
            box-sizing: border-box;
            padding: 4px;
            font-family: inherit;
            font-size: 0.9em;
        }
        .day-text:focus {
            outline: 1px solid #0077cc;
        }
        .day-render {
            flex: 1;
            padding: 4px;
            box-sizing: border-box;
            overflow: auto;
            font-size: 0.9em;
        }
        .empty-cell {
            border: 1px solid transparent;
        }
        .controls-right {
            margin-left: auto;
            display: flex;
            gap: 8px;
        }
        button {
            padding: 4px 10px;
            cursor: pointer;
        }
        input[type="number"], select {
            padding: 3px;
            font-size: 0.9em;
        }
        @media (max-width: 800px) {
            .day-text,
            .day-render {
                font-size: 0.8em;
            }
        }
        .render-mode-active .day-text {
            display: none;
        }
        .render-mode-active .day-render {
            display: block;
        }
        .render-mode-inactive .day-render {
            display: none;
        }
    </style>
    <!-- Markdown-Parser (Client-side) -->
    <script src="https://cdn.jsdelivr.net/npm/marked/marked.min.js"></script>
</head>
<body class="render-mode-inactive">

<div class="topbar">
    <div>
        Year:
        <input type="number" id="yearInput" value="{{ year }}" style="width:70px;">
    </div>
    <div>
        Month:
        <select id="monthSelect">
            {% for m in range(1,13) %}
            <option value="{{ m }}" {% if m == month %}selected{% endif %}>{{ m }}</option>
            {% endfor %}
        </select>
    </div>

    <div class="controls-right">
        <button id="toggleModeBtn">Render anzeigen</button>
        <button id="saveBtn">Save</button>
    </div>
</div>

<div class="title">
    {{ month_name }} {{ year }}
</div>

<div class="wrapper">
    <div class="calendar">
        <!-- Weekday headers -->
        {% for wd in ["Mon","Tue","Wed","Thu","Fri","Sat","Sun"] %}
        <div class="weekday">{{ wd }}</div>
        {% endfor %}

        <!-- Day cells -->
        {% for week in weeks %}
            {% for day in week %}
                {% if day %}
                    <div class="day-cell">
                        <div class="day-number">{{ day }}</div>
                        <textarea class="day-text" data-day="{{ day }}">{{ entries.get(day|string, "") }}</textarea>
                        <div class="day-render" data-day="{{ day }}"></div>
                    </div>
                {% else %}
                    <div class="empty-cell"></div>
                {% endif %}
            {% endfor %}
        {% endfor %}
    </div>
</div>

<script>
    const YEAR = {{ year }};
    const MONTH = {{ month }};

    let isRenderMode = false;

    function collectData() {
        const data = {};
        document.querySelectorAll('.day-text').forEach(function(el) {
            const day = el.getAttribute('data-day');
            data[day] = el.value;
        });
        return data;
    }

    function saveEntries(callback) {
        const payload = {
            year: parseInt(document.getElementById('yearInput').value),
            month: parseInt(document.getElementById('monthSelect').value),
            entries: collectData()
        };

        fetch("/save", {
            method: "POST",
            headers: {
                "Content-Type": "application/json"
            },
            body: JSON.stringify(payload)
        }).then(resp => resp.json())
          .then(data => {
              if (data.status !== "ok") {
                  alert("Error saving data");
              }
              if (callback) callback();
          })
          .catch(err => {
              console.error(err);
              if (callback) callback();
          });
    }

    function updateRenderedMarkdown() {
        document.querySelectorAll('.day-cell').forEach(function(cell) {
            const textarea = cell.querySelector('.day-text');
            const renderDiv = cell.querySelector('.day-render');
            if (!textarea || !renderDiv) return;
            const mdText = textarea.value || "";
            // Markdown -> HTML (client-side)
            renderDiv.innerHTML = marked.parse(mdText);
        });
    }

    function setRenderMode(active) {
        const body = document.body;
        isRenderMode = active;
        if (active) {
            updateRenderedMarkdown();
            body.classList.remove('render-mode-inactive');
            body.classList.add('render-mode-active');
            document.getElementById('toggleModeBtn').textContent = "Markdown anzeigen";
        } else {
            body.classList.remove('render-mode-active');
            body.classList.add('render-mode-inactive');
            document.getElementById('toggleModeBtn').textContent = "Render anzeigen";
        }
    }

    document.getElementById('saveBtn').addEventListener('click', function() {
        saveEntries();
    });

    document.getElementById('toggleModeBtn').addEventListener('click', function() {
        setRenderMode(!isRenderMode);
    });

    document.getElementById('monthSelect').addEventListener('change', function() {
        saveEntries(function() {
            const y = parseInt(document.getElementById('yearInput').value);
            const m = parseInt(document.getElementById('monthSelect').value);
            window.location.href = "/?year=" + y + "&month=" + m;
        });
    });

    document.getElementById('yearInput').addEventListener('change', function() {
        saveEntries(function() {
            const y = parseInt(document.getElementById('yearInput').value);
            const m = parseInt(document.getElementById('monthSelect').value);
            window.location.href = "/?year=" + y + "&month=" + m;
        });
    });
</script>

</body>
</html>
"""

def get_calendar_data(year: int, month: int):
    """
    Returns:
      weeks: list of weeks, each week is list of day numbers or None (Mon..Sun)
    """
    first_weekday, days_in_month = calendar.monthrange(year, month)
    # first_weekday: 0=Mon..6=Sun (Monday-based)

    weeks = []
    day = 1

    # First week
    week = []
    for i in range(7):
        if i < first_weekday:
            week.append(None)
        else:
            week.append(day)
            day += 1
    weeks.append(week)

    # Remaining weeks
    while day <= days_in_month:
        week = []
        for i in range(7):
            if day <= days_in_month:
                week.append(day)
                day += 1
            else:
                week.append(None)
        weeks.append(week)

    return weeks, days_in_month

def get_data_file(year: int, month: int) -> str:
    return os.path.join(DATA_DIR, f"calendar_{year}_{month:02d}.json")

def load_entries(year: int, month: int):
    path = get_data_file(year, month)
    if not os.path.exists(path):
        return {}
    try:
        with open(path, "r", encoding="utf-8") as f:
            return json.load(f)
    except Exception:
        return {}

def save_entries(year: int, month: int, entries: dict):
    path = get_data_file(year, month)
    with open(path, "w", encoding="utf-8") as f:
        json.dump(entries, f, ensure_ascii=False, indent=2)


@app.route("/", methods=["GET"])
def index():
    today = date.today()
    year = request.args.get("year", type=int) or today.year
    month = request.args.get("month", type=int) or today.month

    if year < 1900 or year > 9999:
        year = today.year
    if month < 1 or month > 12:
        month = today.month

    weeks, _ = get_calendar_data(year, month)
    entries = load_entries(year, month)
    month_name = calendar.month_name[month]

    return render_template_string(
        TEMPLATE,
        year=year,
        month=month,
        month_name=month_name,
        weeks=weeks,
        entries=entries
    )


@app.route("/save", methods=["POST"])
def save():
    data = request.get_json(silent=True)
    if not data:
        return jsonify({"status": "error", "message": "No JSON"}), 400

    year = int(data.get("year", 0))
    month = int(data.get("month", 0))
    entries = data.get("entries", {})

    if not (1900 <= year <= 9999 and 1 <= month <= 12):
        return jsonify({"status": "error", "message": "Invalid year/month"}), 400

    entries_str_keys = {str(k): (v or "") for k, v in entries.items()}
    save_entries(year, month, entries_str_keys)
    return jsonify({"status": "ok"})


if __name__ == "__main__":
    app.run(host="127.0.0.1", port=5000, debug=True)
