﻿#NoEnv
#SingleInstance Force
SetWorkingDir, %A_ScriptDir%

; Start with current month
CurrYear  := A_YYYY
CurrMonth := A_MM + 0   ; numeric

Gosub, BuildGui
return


; ============================
; Build GUI for CurrYear/CurrMonth
; ============================
BuildGui:
    Gui, Destroy

    ; --- Date & month ---
    Year  := CurrYear
    Month := CurrMonth
    MonthStr := (Month < 10 ? "0" . Month : Month)
    MonthStart := Year . MonthStr . "01"

    ; Title text
    FormatTime, MonthName, %MonthStart%, MMMM yyyy

    ; --- Days in month ---
    DaysInMonth := 31
    if (Month = 4 or Month = 6 or Month = 9 or Month = 11)
        DaysInMonth := 30
    else if (Month = 2)
    {
        ; Leap year?
        if ( (Mod(Year,400) = 0) or (Mod(Year,4)=0 and Mod(Year,100) != 0) )
            DaysInMonth := 29
        else
            DaysInMonth := 28
    }

    ; Weekday of 1st (AHK: 1=Sun..7=Sat) -> 1=Mon..7=Sun
    FormatTime, DowFirst, %MonthStart%, WDay
    if (DowFirst = 1)
        DowFirst := 7
    else
        DowFirst -= 1

    ; Store globals for resize & save/load
    Global g_DaysInMonth := DaysInMonth
    Global g_DowFirst    := DowFirst

    totalCells := (DowFirst - 1) + DaysInMonth
    NumRows := Ceil(totalCells / 7.0)
    Global g_NumRows := NumRows

    ; Storage file
    Global CalFile := A_ScriptDir . "\calendar_" . Year . "_" . MonthStr . ".txt"

    ; Layout constants
    Global marginX := 10
    Global marginY := 80
    Global gapX    := 5
    Global gapY    := 5

    ; Initial cell sizes (will be recalculated in GuiSize)
    cellW := 140
    cellH := 70
    Global g_cellW := cellW
    Global g_cellH := cellH

    Gui, +Resize
    Gui, Margin, 10, 10
    Gui, Font, s9

    ; --- Year / Month selection ---
    Gui, Add, Text, x10 y10, Year:
    Gui, Add, Edit, x50 y8 w60 vYearInput, %Year%

    Gui, Add, Text, x120 y10, Month:
    ; Dropdown 1–12, reloads month immediately
    Gui, Add, DropDownList, x165 y6 w80 vMonthInput gReloadMonth Choose%Month%, 1|2|3|4|5|6|7|8|9|10|11|12

    ; Title (your edited width kept)
    Gui, Add, Text, vTitleText x10 y35 w550 Center, %MonthName%

    ; Weekday header (Mon–Sun)
    Weekdays := "Mon|Tue|Wed|Thu|Fri|Sat|Sun"
    Loop, Parse, Weekdays, |
    {
        col := A_Index
        x := marginX + (col-1)*(cellW + gapX)
        y := marginY - 20
        Gui, Add, Text, x%x% y%y% w%cellW% Center vHdr%col%, %A_LoopField%
    }

    ; --- Day cells ---
    row := 1
    currentDow := DowFirst
    lastY := marginY

    Loop, %DaysInMonth%
    {
        day := A_Index
        col := currentDow

        x := marginX + (col-1)*(cellW + gapX)
        y := marginY + (row-1)*(cellH + gapY)

        ; Day number label
        Gui, Add, Text, x%x% y%y% w25 Right vL%day%, %day%

        xEdit := x + 25
        wEdit := cellW - 25
        Gui, Add, Edit, x%xEdit% y%y% w%wEdit% h%cellH% vD%day%

        lastY := y

        if (currentDow = 7) {
            currentDow := 1
            row++
        } else {
            currentDow++
        }
    }

    Gosub, LoadMonth

    Gui, Show, w1050 h700, Calendar To-Do – %MonthName%
return


; ============================
; Reload month (year/month changed)
; ============================
ReloadMonth:
    GuiControlGet, NewYear,, YearInput
    GuiControlGet, NewMonth,, MonthInput
    NewYear  += 0
    NewMonth += 0

    if (NewYear < 1900 || NewYear > 9999)
    {
        MsgBox, 48, Error, Invalid year: %NewYear%
        return
    }
    if (NewMonth < 1 || NewMonth > 12)
    {
        MsgBox, 48, Error, Invalid month: %NewMonth%
        return
    }

    ; Save current month before switching
    Gosub, SaveMonth

    CurrYear  := NewYear
    CurrMonth := NewMonth
    Gosub, BuildGui
return


; ============================
; Save current month – multiline-safe
; Format:
; [DAY1]
; line1
; line2
; [ENDDAY]
; ============================
SaveMonth:
    data := ""
    Loop, %g_DaysInMonth%
    {
        day := A_Index
        GuiControlGet, txt,, D%day%
        data .= "[DAY" . day . "]`r`n"
        data .= txt . "`r`n"
        data .= "[ENDDAY]`r`n"
    }
    FileDelete, %CalFile%
    FileAppend, %data%, %CalFile%
return


; ============================
; Load current month (new format)
; ============================
LoadMonth:
    if !FileExist(CalFile)
        return

    currentDay := ""
    buffer := ""

    Loop, Read, %CalFile%
    {
        line := A_LoopReadLine

        ; Start of block?
        if (SubStr(line, 1, 4) = "[DAY")
        {
            ; finalize previous block
            if (currentDay != "" && buffer != "")
            {
                GuiControl,, D%currentDay%, %buffer%
            }
            tmp := SubStr(line, 5)       ; after "DAY"
            StringTrimRight, tmp, tmp, 1 ; remove trailing ]
            currentDay := tmp + 0
            buffer := ""
            continue
        }

        ; End of block?
        if (line = "[ENDDAY]")
        {
            if (currentDay != "")
            {
                GuiControl,, D%currentDay%, %buffer%
            }
            currentDay := ""
            buffer := ""
            continue
        }

        ; Normal line inside block
        if (currentDay != "")
        {
            if (buffer = "")
                buffer := line
            else
                buffer .= "`r`n" . line
        }
    }
return


; ============================
; Resize handler – scale calendar
; ============================
GuiSize:
    if (A_EventInfo = 1) ; minimized
        return

    global marginX, marginY, gapX, gapY, g_DaysInMonth, g_DowFirst, g_NumRows
    global g_cellW, g_cellH

    guiW := A_GuiWidth
    guiH := A_GuiHeight

    availableWidth  := guiW - marginX*2 - gapX*6
    if (availableWidth < 200)
        availableWidth := 200

    rows := g_NumRows
    if (rows < 1)
        rows := 1

    bottomMargin := 40
    availableHeight := guiH - marginY - bottomMargin
    if (availableHeight < 100)
        availableHeight := 100

    cellW := availableWidth / 7.0
    cellH := (availableHeight - gapY*(rows-1)) / rows

    g_cellW := cellW
    g_cellH := cellH

    ; keep title width 550, center it horizontally
    titleW := 550
    titleX := (guiW - titleW) / 2
    if (titleX < 10)
        titleX := 10
    titleY := 35
    GuiControl, Move, TitleText, x%titleX% y%titleY% w%titleW%

    ; Weekday headers
    Loop, 7
    {
        col := A_Index
        x := marginX + (col-1)*(cellW + gapX)
        yHdr := marginY - 20
        GuiControl, Move, Hdr%col%, x%x% y%yHdr% w%cellW%
    }

    ; Day labels & edit fields
    row := 1
    currentDow := g_DowFirst
    Loop, %g_DaysInMonth%
    {
        day := A_Index
        col := currentDow

        x := marginX + (col-1)*(cellW + gapX)
        y := marginY + (row-1)*(cellH + gapY)

        GuiControl, Move, L%day%, x%x% y%y% w25 h%cellH%

        xEdit := x + 25
        wEdit := cellW - 25
        if (wEdit < 20)
            wEdit := 20

        GuiControl, Move, D%day%, x%xEdit% y%y% w%wEdit% h%cellH%

        if (currentDow = 7) {
            currentDow := 1
            row++
        } else {
            currentDow++
        }
    }
return


GuiClose:
    Gosub, SaveMonth
    ExitApp
