#NoEnv
#SingleInstance Force
#Persistent
SetTitleMatchMode, 2
CoordMode, Mouse, Screen
DetectHiddenWindows, Off

global ProfilesIni := A_ScriptDir . "\WindowTools_Profiles.ini"
global TargetWin := 0
global _pick_result := ""

; ============================================================
; PRE-CREATE DYNAMIC SUBMENUS (AHK v1 REQUIREMENT)
; ============================================================
Menu, TrayProf_LoadSub,   Add, (no profiles), TrayProf_LoadChosen
Menu, TrayProf_DeleteSub, Add, (no profiles), TrayProf_DeleteChosen
Menu, CtxProf_LoadSub,    Add, (no profiles), CtxProf_LoadChosen
Menu, CtxProf_DeleteSub,  Add, (no profiles), CtxProf_DeleteChosen

; ============================================================
; PRE-CREATE AND POPULATE ALL STATIC SUBMENUS FIRST
; (AHK v1 requires submenu to exist before linking via :Name)
; ============================================================

; --- Transparency submenu (escape % with `) ---
Menu, TransparencyMenu, Add, 10`% opacity,  Trans10
Menu, TransparencyMenu, Add, 50`% opacity,  Trans50
Menu, TransparencyMenu, Add, 80`% opacity,  Trans80
Menu, TransparencyMenu, Add, 100`% (reset), Trans100

; --- Size submenu (p = height in pixels, 16:9 width) ---
Menu, SizeMenu, Add, 480p (16:9),  Size480
Menu, SizeMenu, Add, 720p (16:9),  Size720
Menu, SizeMenu, Add, 1280p (16:9), Size1280

; --- Placement submenu ---
Menu, PlaceMenu, Add, Top-Left,     PlaceTL
Menu, PlaceMenu, Add, Top-Right,    PlaceTR
Menu, PlaceMenu, Add, Bottom-Left,  PlaceBL
Menu, PlaceMenu, Add, Bottom-Right, PlaceBR

; --- Profiles submenu (context menu) ---
Menu, CtxProfilesMenu, Add, Add this window to profile...,     CtxProf_AddTarget
Menu, CtxProfilesMenu, Add, Update this window in profile...,  CtxProf_UpdateTarget
Menu, CtxProfilesMenu, Add, Remove this window from profile...,    CtxProf_RemoveTarget
Menu, CtxProfilesMenu, Add
Menu, CtxProfilesMenu, Add, Load profile,   :CtxProf_LoadSub
Menu, CtxProfilesMenu, Add, Delete profile, :CtxProf_DeleteSub

; ============================================================
; TRAY MENU
; ============================================================
Menu, Tray, NoStandard
Menu, Tray, Tip, Window Tools (Ctrl+RightClick / Ctrl+Win+T)
Menu, Tray, Add, Help, TrayHelp
Menu, Tray, Add
Menu, Tray, Add, Toggle Always On Top (active), TrayToggleTop
Menu, Tray, Add

; Profiles (tray)
Menu, TrayProfilesMenu, Add, Add active window to profile..., TrayProf_AddActive
Menu, TrayProfilesMenu, Add, Update active window in profile..., TrayProf_UpdateActive
Menu, TrayProfilesMenu, Add, Remove active window from profile..., TrayProf_RemoveActive
Menu, TrayProfilesMenu, Add
Menu, TrayProfilesMenu, Add, Load profile,   :TrayProf_LoadSub
Menu, TrayProfilesMenu, Add, Delete profile, :TrayProf_DeleteSub

Menu, Tray, Add, Profiles, :TrayProfilesMenu
Menu, Tray, Add
Menu, Tray, Add, Exit, TrayExit

; ============================================================
; CONTEXT MENU (now link submenus that already exist)
; ============================================================
Menu, WinTopMenu, Add, Always On Top, ToggleTop
Menu, WinTopMenu, Add
Menu, WinTopMenu, Add, Transparency, :TransparencyMenu
Menu, WinTopMenu, Add, Window size,  :SizeMenu
Menu, WinTopMenu, Add, Placement,    :PlaceMenu
Menu, WinTopMenu, Add
Menu, WinTopMenu, Add, Profiles,     :CtxProfilesMenu

; Build initial dynamic profile submenus
RebuildProfileMenus()

; ============================================================
; HOTKEYS
; ============================================================

; Ctrl + Right Click -> Show menu for window under cursor
^RButton::
    MouseGetPos, mx, my, winID
    if (!winID)
        return

    TargetWin := winID

    ; Update checkmark for Always-On-Top
    WinGet, ExStyle, ExStyle, ahk_id %winID%
    if (ExStyle & 0x8)
        Menu, WinTopMenu, Check, Always On Top
    else
        Menu, WinTopMenu, Uncheck, Always On Top

    ; Refresh profile lists before showing
    RebuildProfileMenus()

    Menu, WinTopMenu, Show
return

; Ctrl + Win + T -> Toggle Always-On-Top for active window
^#t::
    WinGet, activeWin, ID, A
    if (activeWin)
        ToggleTopFor(activeWin)
return

; ============================================================
; CONTEXT MENU HANDLERS
; ============================================================

ToggleTop:
    if (TargetWin)
        ToggleTopFor(TargetWin)
return

; --- Transparency ---
Trans10:
    SetOpacity(TargetWin, 10)
return
Trans50:
    SetOpacity(TargetWin, 50)
return
Trans80:
    SetOpacity(TargetWin, 80)
return
Trans100:
    SetOpacity(TargetWin, 100)
return

; --- Sizes ---
Size480:
    SetSizeP(TargetWin, 480)
return
Size720:
    SetSizeP(TargetWin, 720)
return
Size1280:
    SetSizeP(TargetWin, 1280)
return

; --- Placement ---
PlaceTL:
    PlaceCorner(TargetWin, "TL")
return
PlaceTR:
    PlaceCorner(TargetWin, "TR")
return
PlaceBL:
    PlaceCorner(TargetWin, "BL")
return
PlaceBR:
    PlaceCorner(TargetWin, "BR")
return

; --- Profiles (context) ---
CtxProf_AddTarget:
    if (!TargetWin)
        return
    ProfilePrompt_AddOrUpdate(TargetWin, "add")
return

CtxProf_UpdateTarget:
    if (!TargetWin)
        return
    ProfilePrompt_AddOrUpdate(TargetWin, "update")
return

CtxProf_LoadChosen:
    prof := A_ThisMenuItem
    if (prof != "" && prof != "(no profiles)")
        LoadProfile(prof)
return

CtxProf_DeleteChosen:
    prof := A_ThisMenuItem
    if (prof != "" && prof != "(no profiles)")
        DeleteProfileWithConfirm(prof)
return

CtxProf_RemoveTarget:
    if (!TargetWin)
        return
    ProfilePrompt_Remove(TargetWin)
return

TrayProf_RemoveActive:
    WinGet, activeWin, ID, A
    if (!activeWin)
        return
    ProfilePrompt_Remove(activeWin)
return


; ============================================================
; TRAY HANDLERS
; ============================================================

TrayHelp:
    MsgBox, 64, Window Tools - Help,
    (LTrim
    Usage:
    • Ctrl + Right-Click  → open menu for window under mouse
    • Ctrl + Win + T      → toggle Always-On-Top for active window

    Profiles:
    • Use tray or context menu to add/update/load/delete profiles.
    • Update uses a picker of existing profiles.
    )
return

TrayToggleTop:
    WinGet, activeWin, ID, A
    if (activeWin)
        ToggleTopFor(activeWin)
return

TrayProf_AddActive:
    WinGet, activeWin, ID, A
    if (!activeWin)
        return
    ProfilePrompt_AddOrUpdate(activeWin, "add")
return

TrayProf_UpdateActive:
    WinGet, activeWin, ID, A
    if (!activeWin)
        return
    ProfilePrompt_AddOrUpdate(activeWin, "update")
return

TrayProf_LoadChosen:
    prof := A_ThisMenuItem
    if (prof != "" && prof != "(no profiles)")
        LoadProfile(prof)
return

TrayProf_DeleteChosen:
    prof := A_ThisMenuItem
    if (prof != "" && prof != "(no profiles)")
        DeleteProfileWithConfirm(prof)
return

TrayExit:
    ExitApp
return

; ============================================================
; PROFILE MENU BUILDING
; ============================================================

RebuildProfileMenus() {
    global
    names := GetProfileNames()

    Menu, CtxProf_LoadSub, DeleteAll
    Menu, CtxProf_DeleteSub, DeleteAll
    Menu, TrayProf_LoadSub, DeleteAll
    Menu, TrayProf_DeleteSub, DeleteAll

    if (names.MaxIndex() < 1) {
        Menu, CtxProf_LoadSub, Add, (no profiles), CtxProf_LoadChosen
        Menu, CtxProf_DeleteSub, Add, (no profiles), CtxProf_DeleteChosen
        Menu, TrayProf_LoadSub, Add, (no profiles), TrayProf_LoadChosen
        Menu, TrayProf_DeleteSub, Add, (no profiles), TrayProf_DeleteChosen
        return
    }

    for i, p in names {
        Menu, CtxProf_LoadSub, Add, %p%, CtxProf_LoadChosen
        Menu, CtxProf_DeleteSub, Add, %p%, CtxProf_DeleteChosen
        Menu, TrayProf_LoadSub, Add, %p%, TrayProf_LoadChosen
        Menu, TrayProf_DeleteSub, Add, %p%, TrayProf_DeleteChosen
    }
}

; ============================================================
; PROFILE OPERATIONS (PART 2: UPDATE PICKER)
; ============================================================

ProfilePrompt_AddOrUpdate(winID, mode) {
    global

    names := GetProfileNames()

    if (mode = "update") {
        if (names.MaxIndex() < 1) {
            MsgBox, 48, No profiles, There are no profiles to update yet.
            return
        }
        prof := PickProfileFromList(names, "Update profile", "Select a profile to update:")
        if (prof = "")
            return
    } else {
        defaultName := (names.MaxIndex() >= 1) ? names[1] : ""
        InputBox, prof, Add to profile, Enter profile name:, , 320, 150, , , , , %defaultName%
        if (ErrorLevel)
            return
        prof := Trim(prof)
        if (prof = "")
            return
    }

    AddOrUpdateWindowInProfile(prof, winID)
    RebuildProfileMenus()
    TrayTip, Window Tools, Saved window to profile: %prof%, 2, 1
}

RemoveWindowFromProfile(profileName, winID) {
    global ProfilesIni

    sig := GetWindowSignature(winID)
    if (sig = "")
        return 0

    section := "Profile:" . profileName

    IniRead, raw, %ProfilesIni%, %section%, entries, % ""
    entries := ParseEntries(raw)

    if (!entries.HasKey(sig))
        return 0

    entries.Delete(sig)

    out := SerializeEntries(entries)
    IniWrite, %out%, %ProfilesIni%, %section%, entries
    return 1
}


; --- globals used by picker GUI (AHK v1 label scope limitation)
global _pick_result := ""
global PickChoice := ""

PickProfileFromList(names, title, prompt) {
    global _pick_result, PickChoice

    _pick_result := ""
    PickChoice := ""

    Gui, PickProf:New, +AlwaysOnTop, %title%
    Gui, PickProf:Font, s10
    Gui, PickProf:Add, Text,, %prompt%

    list := ""
    for i, n in names
        list .= (list = "" ? "" : "|") . n

    ; NOTE: control var must be global/static when using g-labels
    Gui, PickProf:Add, ListBox, vPickChoice r10 w340, %list%
    Gui, PickProf:Add, Button, Default gPickProf_OK, OK
    Gui, PickProf:Add, Button, gPickProf_Cancel x+10, Cancel

    Gui, PickProf:Show
    WinWaitClose, %title%
    return _pick_result
}

PickProf_OK:
    global _pick_result, PickChoice
    Gui, PickProf:Submit
    _pick_result := PickChoice
    Gui, PickProf:Destroy
return

PickProf_Cancel:
    global _pick_result
    _pick_result := ""
    Gui, PickProf:Destroy
return


AddOrUpdateWindowInProfile(profileName, winID) {
    global ProfilesIni

    sig := GetWindowSignature(winID)
    if (sig = "")
        return

    state := GetWindowState(winID)
    if (state = "")
        return

    section := "Profile:" . profileName

    IniRead, raw, %ProfilesIni%, %section%, entries, % ""
    entries := ParseEntries(raw)

    entries[sig] := state  ; overwrite / update

    out := SerializeEntries(entries)
    IniWrite, %out%, %ProfilesIni%, %section%, entries

    EnsureProfileListed(profileName)
}

LoadProfile(profileName) {
    global ProfilesIni

    section := "Profile:" . profileName
    IniRead, raw, %ProfilesIni%, %section%, entries, % ""
    if (raw = "") {
        TrayTip, Window Tools, Profile is empty: %profileName%, 2, 1
        return
    }

    entries := ParseEntries(raw)
    applied := 0

    for sig, state in entries {
        winID := FindWindowBySignature(sig)
        if (winID) {
            ApplyWindowState(winID, state)
            applied++
        }
    }

    TrayTip, Window Tools, Loaded "%profileName%": %applied% window(s) applied, 2, 1
}

DeleteProfileWithConfirm(profileName) {
    global ProfilesIni
    MsgBox, 49, Delete profile?, Delete profile "%profileName%"?`n(This cannot be undone.)
    IfMsgBox, Cancel
        return

    IniDelete, %ProfilesIni%, % "Profile:" . profileName
    RemoveProfileListed(profileName)
    RebuildProfileMenus()
    TrayTip, Window Tools, Deleted profile: %profileName%, 2, 1
}

EnsureProfileListed(profileName) {
    global ProfilesIni
    IniRead, list, %ProfilesIni%, Profiles, names, % ""
    arr := ParseCsv(list)

    for i, n in arr
        if (n = profileName)
            return

    arr.Push(profileName)
    IniWrite, % JoinCsv(arr), %ProfilesIni%, Profiles, names
}

RemoveProfileListed(profileName) {
    global ProfilesIni
    IniRead, list, %ProfilesIni%, Profiles, names, % ""
    arr := ParseCsv(list)

    out := []
    for i, n in arr
        if (n != "" && n != profileName)
            out.Push(n)

    IniWrite, % JoinCsv(out), %ProfilesIni%, Profiles, names
}

GetProfileNames() {
    global ProfilesIni
    IniRead, list, %ProfilesIni%, Profiles, names, % ""
    arr := ParseCsv(list)

    out := []
    for i, n in arr
        if (Trim(n) != "")
            out.Push(Trim(n))
    return out
}

ProfilePrompt_Remove(winID) {
    global
    names := GetProfileNames()
    if (names.MaxIndex() < 1) {
        MsgBox, 48, No profiles, There are no profiles yet.
        return
    }

    prof := PickProfileFromList(names, "Remove from profile", "Select a profile to remove this window from:")
    if (prof = "")
        return

    removed := RemoveWindowFromProfile(prof, winID)
    RebuildProfileMenus()

    if (removed)
        TrayTip, Window Tools, Removed window from profile: %prof%, 2, 1
    else
        TrayTip, Window Tools, Window not found in profile: %prof%, 2, 1
}


; ============================================================
; WINDOW SIGNATURE + STATE (PART 1: MULTI-MONITOR AWARE)
; ============================================================

GetMonitorIndexForPoint(x, y) {
    SysGet, monCount, MonitorCount
    Loop, %monCount% {
        SysGet, wa, MonitorWorkArea, %A_Index%
        if (x >= waLeft && x <= waRight && y >= waTop && y <= waBottom)
            return A_Index
    }
    return 1
}

GetWindowSignature(winID) {
    WinGet, exe, ProcessName, ahk_id %winID%
    WinGetClass, cls, ahk_id %winID%
    WinGetTitle, title, ahk_id %winID%

    exe := Trim(exe), cls := Trim(cls), title := Trim(title)
    if (exe = "" && cls = "" && title = "")
        return ""

    return exe . "|" . cls . "|" . title
}

FindWindowBySignature(sig) {
    parts := StrSplit(sig, "|")
    if (parts.MaxIndex() < 3)
        return 0

    exe := parts[1], cls := parts[2], title := parts[3]

    WinGet, list, List
    Loop, %list% {
        id := list%A_Index%
        WinGet, e, ProcessName, ahk_id %id%
        WinGetClass, c, ahk_id %id%
        WinGetTitle, t, ahk_id %id%

        if (e = exe && c = cls && t = title)
            return id
    }
    return 0
}

GetWindowState(winID) {
    ; New state format:
    ; monIndex, relX, relY, w, h, opacityPct, topmost(0/1)

    ; Restore first to get reliable coords (fixes "needs second load")
    WinGet, mm, MinMax, ahk_id %winID%
    if (mm != 0)
        WinRestore, ahk_id %winID%

    WinGetPos, x, y, w, h, ahk_id %winID%

    cx := x + (w // 2)
    cy := y + (h // 2)
    mon := GetMonitorIndexForPoint(cx, cy)

    SysGet, wa, MonitorWorkArea, %mon%
    relX := x - waLeft
    relY := y - waTop

    WinGet, trans, Transparent, ahk_id %winID%
    if (trans = "")
        pct := 100
    else {
        pct := Round((trans / 255.0) * 100)
        if (pct < 1)
            pct := 1
        if (pct > 100)
            pct := 100
    }

    WinGet, ExStyle, ExStyle, ahk_id %winID%
    top := (ExStyle & 0x8) ? 1 : 0

    return mon . "," . relX . "," . relY . "," . w . "," . h . "," . pct . "," . top
}

ApplyWindowState(winID, state) {
    parts := StrSplit(state, ",")

    ; Backward compatibility: old format x,y,w,h,pct,top
    if (parts.MaxIndex() = 6) {
        x := parts[1], y := parts[2], w := parts[3], h := parts[4], pct := parts[5], top := parts[6]

        WinGet, mm, MinMax, ahk_id %winID%
        if (mm != 0)
            WinRestore, ahk_id %winID%

        WinMove, ahk_id %winID%, , %x%, %y%, %w%, %h%
        Sleep, 30
        WinMove, ahk_id %winID%, , %x%, %y%, %w%, %h%
        SetOpacity(winID, pct)

        if (top = 1)
            WinSet, AlwaysOnTop, On, ahk_id %winID%
        else
            WinSet, AlwaysOnTop, Off, ahk_id %winID%
        return
    }

    ; New format: mon,relX,relY,w,h,pct,top
    if (parts.MaxIndex() < 7)
        return

    mon := parts[1], relX := parts[2], relY := parts[3]
    w := parts[4], h := parts[5], pct := parts[6], top := parts[7]

    SysGet, monCount, MonitorCount
    if (mon < 1 || mon > monCount)
        mon := 1

    SysGet, wa, MonitorWorkArea, %mon%
    x := waLeft + relX
    y := waTop  + relY

    WinGet, mm, MinMax, ahk_id %winID%
    if (mm != 0)
        WinRestore, ahk_id %winID%

    ; Apply twice for reliability
    WinMove, ahk_id %winID%, , %x%, %y%, %w%, %h%
    Sleep, 30
    WinMove, ahk_id %winID%, , %x%, %y%, %w%, %h%

    SetOpacity(winID, pct)

    if (top = 1)
        WinSet, AlwaysOnTop, On, ahk_id %winID%
    else
        WinSet, AlwaysOnTop, Off, ahk_id %winID%
}

; ============================================================
; STORAGE (INI serialization)
; ============================================================

ParseEntries(raw) {
    entries := {}
    raw := Trim(raw)
    if (raw = "" || raw = "ERROR")
        return entries

    pairs := StrSplit(raw, ";;")
    for i, p in pairs {
        if (p = "")
            continue
        eq := InStr(p, "=")
        if (!eq)
            continue
        k := SubStr(p, 1, eq-1)
        v := SubStr(p, eq+1)
        if (k != "")
            entries[k] := v
    }
    return entries
}


SerializeEntries(entries) {
    out := ""
    for k, v in entries {
        ; Minimal escaping to avoid delimiters
        k2 := StrReplace(k, ";;", "·")
        k2 := StrReplace(k2, "=",  "≡")
        v2 := StrReplace(v, ";;", "·")
        v2 := StrReplace(v2, "=",  "≡")
        out .= k2 . "=" . v2 . ";;"
    }
    return out
}

ParseCsv(s) {
    arr := []
    s := Trim(s)
    if (s = "" || s = "ERROR")
        return arr
    parts := StrSplit(s, ",")
    for i, p in parts
        arr.Push(Trim(p))
    return arr
}


JoinCsv(arr) {
    out := ""
    for i, p in arr {
        if (p = "")
            continue
        if (out != "")
            out .= ","
        out .= p
    }
    return out
}

; ============================================================
; WINDOW OPS
; ============================================================

ToggleTopFor(winID) {
    WinGet, ExStyle, ExStyle, ahk_id %winID%
    if (ExStyle & 0x8)
        WinSet, AlwaysOnTop, Off, ahk_id %winID%
    else
        WinSet, AlwaysOnTop, On,  ahk_id %winID%
}

SetOpacity(winID, pct) {
    if (!winID)
        return
    if (pct < 1)
        pct := 1
    if (pct > 100)
        pct := 100
    alpha := Round(255 * (pct / 100.0))
    WinSet, Transparent, %alpha%, ahk_id %winID%
}

SetSizeP(winID, height) {
    if (!winID)
        return
    width := Round(height * 16 / 9)
    WinGetPos, x, y, w, h, ahk_id %winID%
    WinMove, ahk_id %winID%, , %x%, %y%, %width%, %height%
}

PlaceCorner(winID, corner) {
    if (!winID)
        return

    WinGetPos, wx, wy, ww, wh, ahk_id %winID%
    cx := wx + (ww // 2)
    cy := wy + (wh // 2)

    SysGet, monCount, MonitorCount
    targetMon := 1
    Loop, %monCount% {
        SysGet, wa, MonitorWorkArea, %A_Index%
        if (cx >= waLeft && cx <= waRight && cy >= waTop && cy <= waBottom) {
            targetMon := A_Index
            break
        }
    }
    SysGet, wa, MonitorWorkArea, %targetMon%

    margin := 10
    if (corner = "TL") {
        nx := waLeft + margin
        ny := waTop  + margin
    } else if (corner = "TR") {
        nx := waRight - ww - margin
        ny := waTop   + margin
    } else if (corner = "BL") {
        nx := waLeft + margin
        ny := waBottom - wh - margin
    } else {
        nx := waRight - ww - margin
        ny := waBottom - wh - margin
    }
    WinMove, ahk_id %winID%, , %nx%, %ny%
}
